  /*
   *  Object %CRYS_HASH.c    : %
   *  State           :  %state%
   *  Creation date   :  Wed Nov 17 17:25:55 2004
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version CRYS_HASH.c#1:csrc:6
   *  \author adams
   *  \remarks Copyright (C) 2004 by Discretix Technologies Ltd.
   *           All Rights reserved
   */



/************* Include Files ****************/

/* .............. CRYS level includes ................. */

#include "DX_VOS_Mem.h"

#include "CRYS_HASH_error.h"
#include "CRYS.h"
#include "dx_hw_defs.h"
#include "crys_host_op_code.h"


/* .............. LLF level includes ................. */

 
/************************ Defines ***************************************/

/* canceling the PC-lint warning:
   Use of goto is deprecated */
/*lint --e{801} */

/************************ Enums *****************************************/

/************************ macros ****************************************/

/************************ Global Data ***********************************/

/************* Private function prototype *******************************/

/************************ Public Functions ******************************/

/**
 * This function initializes the HASH machine on the CRYS level.
 *
 * This function allocates and initializes the HASH Context .
 * The function receives as input a pointer to store the context handle to HASH Context , 
 * it initializes the 
 * HASH Context with the cryptographic attributes that are needed for 
 * the HASH block operation ( initialize H's value for the HASH algorithm ).
 *
 * The function flow:
 *
 * 1) checking the validity of the arguments - returnes an error on an illegal argument case.
 * 2) Aquiring the working context from the CCM manager.
 * 3) Initializing the context with the parameters passed by the user and with the init values
 *    of the HASH.
 * 4) loading the user tag to the context.
 * 5) release the CCM context.
 * 
 * @param[in] ContextID_ptr - a pointer to the HASH context buffer allocated by the user that
 *                       is used for the HASH machine operation.
 *
 * @param[in] OperationMode - The operation mode : MD5 or SHA1.
 *
 * @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 *      
 */
CEXPORT_C CRYSError_t CRYS_HASH_Init( CRYS_HASHUserContext_t     *ContextID_ptr,
                                      CRYS_HASH_OperationMode_t  OperationMode)
{
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_INIT_OP_CODE, 2,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 OperationMode );

}/* END OF CRYS_HASH_Init */                           

/**
 * This function process a block of data via the HASH Hardware.
 * The function receives as input an handle to the  HASH Context, that was initialized before
 * by an CRYS_HASH_Init function or by other CRYS_HASH_Update function. \n
 * The function Sets the hardware with the last H's value that where stored in
 * the CRYS HASH context and then process the data block using the hardware 
 * and in the end of the process stores in the HASH context the H's value 
 * HASH Context with the cryptographic attributes that are needed for 
 * the HASH block operation ( initialize H's value for the HASH algorithm ).
 * This function is used in cases not all the data is arrange in one 
 * continues buffer . \n
 *
 * The function flow:
 *
 * 1) checking the parameters validty if there is an error the function shall exit with an error code. 
 * 2) Aquiring the working context from the CCM manager.
 * 3) If there isnt enouth data in the previous update data buff in the context plus the received data
 *    load it to the context buffer and exit the function.
 * 4) fill the previous update data buffer to contain an entire block. 
 * 5) Calling the hardware low level function to execute the update.
 * 6) fill the previous update data buffer with the data not processed at the end of the received data.
 * 7) release the CCM context.
 * 
 * @param[in] ContextID_ptr - a pointer to the HASH context buffer allocated by the user that
 *                       is used for the HASH machine operation.
 *
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the HASH. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 * 
 * @param[in] DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
 *                         multiple of HASH_BLOCK_SIZE for used HASH mode.
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_HASH_error.h
 *
 */

CEXPORT_C CRYSError_t CRYS_HASH_Update(CRYS_HASHUserContext_t  *ContextID_ptr,
                                       DxUint8_t               *DataIn_ptr,
                                       DxUint32_t               DataInSize )
{                                              
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_UPDATE_OP_CODE, 2,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 DataIn_ptr, DataInSize );
    
}/* END OF CRYS_HASH_Update */ 

/**
 * This function finalize the hashing process of data block.
 * The function receives as input an handle to the HASH Context , that was initialized before
 * by an CRYS_HASH_Init function or by CRYS_HASH_Update function.
 * The function "adds" an header to the data block as the specific hash standard 
 * specifics , then it loads the hardware and reads the final message digest.
 *
 *  the function flow:
 *
 * 1) checking the parameters validty if there is an error the function shall exit with an error code. 
 * 2) Aquiring the working context from the CCM manager.
 * 3) Setting the padding buffer to load.
 * 4) Calling the hardware low level function to execute the finish.
 * 5) fill the previous update data buffer with the data not processed at the end of the received data.
 * 6) release the CCM context.
 *
 *  @param[in] ContextID_ptr - a pointer to the HASH context buffer allocated by the user that
 *                       is used for the HASH machine operation.
 *
 *  @retval HashResultBuff a pointer to the target buffer where the 
 *                       HASE result stored in the context is loaded to.
 *
 *  @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 */

CEXPORT_C CRYSError_t CRYS_HASH_Finish( CRYS_HASHUserContext_t  *ContextID_ptr ,
                                        CRYS_HASH_Result_t       HashResultBuff )
{  
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_FINISH_OP_CODE, 2,
	   							 /* PARAMETERS: */
								 ContextID_ptr,
								 HashResultBuff );
    
}/* END OF CRYS_HASH_Finish */

/**
 * @brief This function is a service function that frees the context if the operation has
 *        failed. 
 *
 *        The function executes the following major steps:
 *
 *        1. Checks the validity of all of the inputs of the function. 
 *           If one of the received parameters is not valid it shall return an error.
 *
 *           The major checkers that are run over the received parameters:
 *           - verifying that the context pointer is not DX_NULL (*ContextID_ptr). 
 *        2. Clearing the users context.
 *        3. Exit the handler with the OK code.
 *      
 *
 * @param[in] ContextID_ptr - a pointer to the HASH context buffer allocated by the user that
 *                       is used for the HASH machine operation. this should be the same context that was
 *                       used on the previous call of this session.
 *
 *
 * @return CRYSError_t - On success CRYS_OK is returned, on failure a
 *                        value MODULE_* CRYS_HASH_error.h
 */
CEXPORT_C CRYSError_t  CRYS_HASH_Free(CRYS_HASHUserContext_t  *ContextID_ptr )
{
  /* The return error identifier */
  CRYSError_t Error;
  
  /*---------------------------
      CODE
  -----------------------------*/
 
  /* ............... local initializations .............................. */
   
  Error = CRYS_OK;
                      
  /* ............... checking the parameters validity ................... */
  
  /* if the users context ID pointer is DX_NULL return an error */
  if( ContextID_ptr == DX_NULL )
  {
    Error = CRYS_HASH_INVALID_USER_CONTEXT_POINTER_ERROR;
    goto end_function;
  }

  /* .............. clearing the users context .......................... */
  DX_VOS_MemSet( ContextID_ptr , 0 , sizeof(CRYS_HASHUserContext_t) );   

   /* ................. end of function ..................................... */
   /* ----------------------------------------------------------------------- */

end_function:
       
  return Error;
   
}/* END OF CRYS_HASH_Free */
                     
/**
 * This function provide HASH function to process one buffer of data.
 * The function allocates an internal HASH Context , it initializes the 
 * HASH Context with the cryptographic attributes that are needed for 
 * the HASH block operation ( initialize H's value for the HASH algorithm ).
 * Then the function loads the Hardware with the initializing values and after 
 * that process the data block using the hardware to do hash .
 * At the end the function return the message digest of the data buffer .
 *
 *
 * @param[in] OperationMode - The operation mode : MD5 or SHA1.
 * 
 * @param[in] DataIn_ptr - The pointer to the buffer of the input data to the HASH. The pointer does 
 *                         not need to be aligned. On CSI input mode the pointer must be equal to
 *                         value (0xFFFFFFFC | DataInAlignment). 
 * 
 * @param[in] DataInSize - The size of the data to be hashed in bytes. On CSI data transfer mode the size must  
 *                         multiple of HASH_BLOCK_SIZE for used HASH mode.
 *
 * @retval HashResultBuff a pointer to the target buffer where the 
 *                      HASE result stored in the context is loaded to.
 *
 * @return CRYSError_t on success the function returns CRYS_OK else non ZERO error.
 *
 */
CEXPORT_C CRYSError_t CRYS_HASH( CRYS_HASH_OperationMode_t  OperationMode,
                                 DxUint8_t                 *DataIn_ptr,
                                 DxUint32_t                 DataSize,
                                 CRYS_HASH_Result_t         HashResultBuff )
{    
 	return DX_HW_Dispatcher_ptr( DX_SEP_HOST_SEP_PROTOCOL_HOST_HASH_OP_CODE, 4,
	   							 /* PARAMETERS: */
								 OperationMode,
								 DataIn_ptr, DataSize,
								 HashResultBuff );
    
}/* END OF CRYS_HASH */   
